// gs_directdraw.h

#ifndef GS_DIRECTDRAW_H
#define GS_DIRECTDRAW_H

// Includes
#include "ddraw.h"
#include <fstream>
//#include <direct.h>
#include <cstdio>


#include "gs_common.h"
#include "gs_debug.h"
#include "gs_winstandard.h"

// Global defines
#define COLOR_BITS			16
#define MAX_COLORS			65535
#define MAX_BLEND			255
#define MAX_CLIP_RECTS		5

// Defines to be used for FontObject

// Fonts (standard ones on all versions of Windows)
#define FONT_VERDANA			"Verdana"
#define FONT_TIMES_NEW_ROMAN	"Times New Roman"
#define FONT_COMIC_SANS_MS		"Comic Sans MS"
#define FONT_COURIER			"Courier"
#define FONT_COURIER_NEW		"Courier New"
#define FONT_ARIAL				"Arial"
#define FONT_SYMBOL 			"Symbol"


// Colors
// (NOTE, format = 0x00BBGGRR)
#define TEXT_WHITE			0x00FFFFFF
#define TEXT_BLACK			0x00000000
#define TEXT_RED			0x000000FF
#define TEXT_GREEN			0x0000FF00
#define TEXT_BLUE			0x00FF0000




// Struct used to hold information of enumerated display devices
typedef struct
{
	LPGUID lpGuid;
	char DriverDescription[MAX_STRING_LENGTH<<1];
	char DriverName[MAX_STRING_LENGTH<<1];
} tDisplayDeviceInfo;

/*
// String used to hold clipper information for each surface
typedef  struct
{
	RGNDATAHEADER rdh;
	RECT Rects[MAX_CLIP_RECTS];
} tClipList;
*/

//**********************************************************************
class DirectDrawObject
{
	public:

		// Public member functions
		DirectDrawObject(void); // Constructor
		~DirectDrawObject(void); // Destructor

		DirectDrawObject(const DirectDrawObject &S); // Copy constructor
		DirectDrawObject& operator=(const DirectDrawObject &S); // Member assignment

		int EnumerateDevices(void); // Fills up an internal list of display devices, returns how many
		char* GetDeviceDescription(int device_num);  // Returns the name of device device_num

		// Member functions to create and destroy the main window
		HWND CreateWin(HINSTANCE hInstance, LPCSTR caption = "Guardian Soft", int width = CW_USEDEFAULT, int height = CW_USEDEFAULT);
		int DestroyWin(void); // Should be called at the end of main (i.e. "return DD.DestroyWindow();")
		WindowObject* GetWindowObject(void); // Returns a pointer to the window object (or NULL if none is defined)

		// Overloaded function that takes a pointer to a WindowObject
		HWND CreateWin(WindowObject *W);

		// Uses an already existing window
		HWND UseWin(HWND hWnd, int width, int height, DWORD win_style, DWORD win_style_ex);

		// Functions to set and switch the screen mode. Call SetMode() at initialization time, 
		// and call SwitchMode() during runtime.
		void SetMode(int display_mode=FULLSCREEN, int display_device=NULL);
		void SwitchMode(void); // Switches to WINDOWED or FULLSCREEN mode

		void SetGraphicsDirectory(char relative_path[]); // Set the graphics directory for the Load functions

		// Functions called by GSWinUpdate() (defined in gs_helper.h)
		void LostWinFocus(void); // Called when the window loses focus
		bool GotWinFocus(void); // Called when the window regains the focus

	private:

		int ID; // The Object ID
};

//**********************************************************************
class FontObject
{
	friend class SurfaceObject; // Now SurfaceObject can access FontObject's private data

	friend class GenericSurfaceObject;

	public:

		FontObject(void); // Constructor
		~FontObject(void); // Destructor

		// Call this to create the font object; you must specify at least the font
		// name, and the height
		void Create(char font_name[], int height, COLORREF text_color=TEXT_WHITE, 
					COLORREF bk_color=TEXT_BLACK, bool bk_transparent=true,
					bool bold=false, bool italic=false, bool underline=false, bool strikeout=false);

		void SetFont(char font_name[]); // To change to a new font
		void SetHeight(int height); // To change to a new height
		void SetColors(COLORREF text_color, COLORREF bk_color); // To change the colors
		void SetBkTransparent(bool bk_transparent); // True for transparent background
		void SetAttributes(bool bold=false, bool italic=false, bool underline=false, bool strikeout=false);


	private:

		// Private member functions
		bool IsValid(void); // Returns true if the FontObject has been created properly
		void UpdateFont(void); // Updates the font object with the current state

		// Private Data members
		HFONT hFont; // Handle to the font object

		int Height; // Height of the font
		char *FontName; // The font's name
		COLORREF TextColor, TextBkColor; // Text and Text background colors
		bool TextBkTransparent; // Is background transparent

		bool Bold, Italic, Underline, Strikeout; // Attributes of the font

		int ID; // The Object ID
};


//**********************************************************************
class SurfaceObject
{
	public:

		// Public member functions
		SurfaceObject(void); // Constructor (for Primary and Memory surfaces)
		~SurfaceObject(void); // Destructor

		SurfaceObject(const SurfaceObject &S); // Copy constructor
		SurfaceObject& operator=(const SurfaceObject &S); // Member assignment

		void Create(int width, int height); // Creates regular surfaces (memory)
		void CreatePrimary(HWND different_hwnd=NULL); // Creates Primary surface (VGA)
		void Destroy(void); // Manual Destructor

		void Display(void); // Displays Primary surface (only) to screen

		void LoadRAW(char filename[], int width, int height, int offset=0); // Load RAW file onto Surface (same size)
		void LoadBMP(char filename[]);			  // Load BMP file onto Surface (same size)

		void SetFontObject(FontObject &font_object); // Use this to pass a new font object to be used by Text()
		void Text(int x, int y, char string[], ...); // Writes plain text at (x,y) on the surface - good for debugging

		// Some mutators used to set certain parameters for the surface
		void SetSpriteColor(unsigned short color);		// Set the sprite color
		void SetSpriteColor(unsigned char red, unsigned char green, unsigned char blue);

		void SetBlendLevel(unsigned char level);		// Set the blend level (0 to 255)
		void SetTiled(int tile_width, int tile_height); // Set surface as tiled (to use CopyTile() methods)
		
		// Clipping functions (should be used in this order)
		void AddClipRect(void); // Defines a clipping rectangle equal to the size of the surface

		// *** NOT RECOMMENDED YET - BUGGY ***
		void AddClipRect(int x, int y, int x2, int y2); // Can define the corners
		void AddClipRectSize(int x, int y, int width, int height); // Can define the size

		void AttachClipper(void); // Attaches the clipping rectangles defined in AddClipRect[Size] to surface
		void DetachClipper(void); // Simply detaches the clipper (can be reattached using AttachClipper())
		
		void RemoveClipper(void); // Removes the clipper from the surface completely

		// The most important methods, used to set the copy rectangle on the surface
		void SetRect(int x, int y); // Sets the upper left corner of the rect (and adjusts the lower right corner)
		void SetRect(int x, int y, int x2, int y2); // Set copy rectangle (with lower-right coordinates)
		void SetRectSize(int x, int y, int width, int height); // Set copy rectangle (using width and height)
		void ResetRect(void); // Resets the copy rectangle to the full size of the surface

		// Corresponding investigator methods
		unsigned short GetSpriteColor(void); // Returns Sprite color
		unsigned char GetBlendLevel(void);	 // Returns Blend level
		bool GetTiled(void);				 // Returns true if surface is tiled

		void GetRect(int &x, int &y);
		void GetRect(int &x, int &y, int &x2, int &y2);
		void GetRectSize(int &x, int &y, int &width, int &height);

		void GetSurfSize(int &width, int &height); // Gets width and height of the surface
		int GetSurfWidth(void); // Returns the width of the surface
		int GetSurfHeight(void); // Returns the height of the surface
		int GetSurfPitch(void); // Returns the actual width of the surface (not the logical width)


		// Following methods are affected by above Set functions

		void Fill(unsigned short color); // Fill copy rectangle with color
		void Fill(unsigned char red, unsigned char green, unsigned char blue);


		// CopyFast() only works if the source and destination are the same size and if the destination
		// surface does not have a clipper attached to it
		void CopyFast(SurfaceObject &Source);

		// Regular copy functions - Use values set by the SetRect() functions to
		// determine source and destination copy areas
		void Copy(SurfaceObject &Source);
		void Copy_Sprite(SurfaceObject &Source);

		void Copy_Blend(SurfaceObject &Source);
		void Copy_BlendSprite(SurfaceObject &Source);

		void Copy_Scale(SurfaceObject &Source);
		void Copy_ScaleSprite(SurfaceObject &Source);

		void Copy_HFlip(SurfaceObject &Source);
		void Copy_VFlip(SurfaceObject &Source);

		void Copy_HFlipSprite(SurfaceObject &Source);
		void Copy_VFlipSprite(SurfaceObject &Source);


		// Tile copy functions (Surface must be tiled using SetTiled())
		void CopyTile(SurfaceObject &Source, int tile);
		void CopyTile_Sprite(SurfaceObject &Source, int tile);

		void CopyTile_Blend(SurfaceObject &Source, int tile);
		void CopyTile_BlendSprite(SurfaceObject &Source, int tile);

		void CopyTile_Scale(SurfaceObject &Source, int tile);
		void CopyTile_ScaleSprite(SurfaceObject &Source, int tile);

		void CopyTile_HFlip(SurfaceObject &Source, int tile);
		void CopyTile_VFlip(SurfaceObject &Source, int tile);

		void CopyTile_HFlipSprite(SurfaceObject &Source, int tile);
		void CopyTile_VFlipSprite(SurfaceObject &Source, int tile);

		unsigned short* Lock(void); // Returns a pointer to the surface
		void Unlock(void); // Make sure to call Unlock() as soon as possible after calling Lock()

	private:
		// Private data members
		LPDIRECTDRAWSURFACE4 Surface; // Surface represented by this class
		int SurfWidth, SurfHeight; // Width and Height of Surface

		RECT CopyRect; // Defines area to copy
		int Width, Height; // Width and Height in copy rectangle

		RECT *TileRects; // May contain rect's for each tile (if SetTiled() is called)
		int Tile_Width, Tile_Height; // Width and Height of tiles

		LPDIRECTDRAWCLIPPER Clipper; // Clipper object
		int NumClipRects; // Number of clipping rectangles defined
//		tClipList *ClipList; // The clip list

		unsigned char BlendLevel; // Blend level
		unsigned short SpriteColor; // Sprite color
		long Pitch; // Use instead of SurfWidth when calling Lock()

		DDSURFACEDESC2 ddsd; // Surface description structure
		unsigned char Flags; // Contains information about the surface

		FontObject *FontObjectPtr; // Pointer to a FontObject

		int ID; // The Object ID

		// Private member functions		

		// Construction member function called by Constructor, Copy Constructor, and member assignment
		void Construct(void);

		// Hand-coded blending function used by all the "Blend" Copy functions
		void Blend(SurfaceObject &Source, bool spriteBlend, int destOffset, int sourceOffset, int width, int height);
			
		// Defined inline for speed...
		void AdjustDestRect(SurfaceObject &Source) 
		{
			CopyRect.right = CopyRect.left + Source.Width;
			CopyRect.bottom = CopyRect.top + Source.Height;
		}

		void AdjustDestRectTile(SurfaceObject &Source, int tile)
		{
			CopyRect.right = CopyRect.left + Source.Tile_Width;
			CopyRect.bottom = CopyRect.top + Source.Tile_Height;
		}
};

#endif