// gs_surface.h

///////////////////////////////////////////////////////////////////////////////
// This file contains the interfaces for the different surface objects that can
// be created. The class hierarchy looks as follows:
//
//
//                        GenericSurfaceObject
//                          |              |
//                          |              |
//                         /         DisplaySurfaceObject
//                       /              |              |
//                      |               |              |
//    GenericSurfaceObject   *PrimarySurfaceObject  WindowSurfaceObject
//
//
// Only the bottom-most classes can be instantiated.
// *PrimarySurfaceObject is a singleton - only one instance of it can exist,
// and to get it, you must call PrimarySurfaceObject::Instance().
///////////////////////////////////////////////////////////////////////////////

#ifndef GS_SURFACE_H
#define GS_SURFACE_H

#include "gs_common.h"
#include "gs_directdraw.h"
#include "gs_winstandard.h"
#include "gs_font.h"

// Defines
#define MAX_COLORS			65535
#define MAX_BLEND			255
#define MAX_CLIP_RECTS		5

// Structure which holds pixel RGB masks
typedef struct _RGBMASK
{
	unsigned long rgbRed; // red component
	unsigned long rgbGreen; // green component
	unsigned long rgbBlue; // blue component
} RGBMASK;

// Structure which holds screen surface info (5,6,5 or 5,5,5 and masking)
typedef struct _RGB16
{
	RGBQUAD depth;
	RGBQUAD Amount;
	RGBQUAD Position;
	RGBMASK Mask;
} RGB16;

typedef enum {REGULAR_SURF, PRIMARY_REGULAR_SURF, PRIMARY_BACKBUFFER_SURF} tSurfType;

//**********************************************************************
// The base class. Cannot be instatiated
class GenericSurfaceObject
{
	public:

		// Only children can construct/destruct GenericSurfaceObjects
		GenericSurfaceObject();				// Constructor
		virtual ~GenericSurfaceObject();	// Destructor
		
		GenericSurfaceObject(const GenericSurfaceObject &S);			// Copy constructor
		GenericSurfaceObject& operator=(const GenericSurfaceObject &S);	// Assignment operator	

		// Public member functions
		bool IsCreated(); // Returns true if the object is created

		// Image loading functions - these also create the surface based on the file size.
		void LoadRAW(char filename[], int width, int height, int offset=0);	// Load RAW file onto Surface (same size)
		void LoadBMP(char filename[]);										// Load BMP file onto Surface (same size)

		// Font and text output functions
		void SetFontObject(FontObject &font_object); // Use this to pass a new font object to be used by Text()
		void Text(int x, int y, char string[], ...); // Writes plain text at (x,y) on the surface - good for debugging

		// Mutators used to set certain parameters for the surface
		void SetSpriteColor(unsigned short color);		// Set the sprite color
		void SetSpriteColor(unsigned char red, unsigned char green, unsigned char blue);
		void SetBlendLevel(unsigned char level);		// Set the blend level (0 to 255)
		void SetTiled(int tile_width, int tile_height); // Set surface as tiled (to use CopyTile() methods)
		
		// Clipping functions
		void AddClipRect(); // Defines a clipping rectangle equal to the size of the surface
		void AddClipRect(int x, int y, int x2, int y2); // Can define the corners
		void AddClipRectSize(int x, int y, int width, int height); // Can define the size
		void DeleteClipRects(); // Deletes all the clipping rectangles

		void AttachClipper(); // Attaches the clipping rectangles defined in AddClipRect[Size] to surface
		void DetachClipper(); // Simply detaches the clipper (can be reattached using AttachClipper()


		// The most important methods: used to set the copy rectangle on the surface
		void SetRect(int x, int y); // Sets the upper left corner of the rect (and adjusts the lower right corner)
		void SetRect(int x, int y, int x2, int y2); // Set copy rectangle (with lower-right coordinates)
		void SetRectSize(int x, int y, int width, int height); // Set copy rectangle (using width and height)
		void ResetRect(); // Resets the copy rectangle to the full size of the surface

		// Corresponding inspectors
		unsigned short GetSpriteColor(); // Returns Sprite color
		unsigned char GetBlendLevel();	 // Returns Blend level
		bool GetTiled();				 // Returns true if surface is tiled

		void GetRect(int &x, int &y);
		void GetRect(int &x, int &y, int &x2, int &y2);
		void GetRectSize(int &x, int &y, int &width, int &height);

		void GetSurfSize(int &width, int &height); // Gets width and height of the surface
		int GetSurfWidth(); // Returns the width of the surface
		int GetSurfHeight(); // Returns the height of the surface
		int GetSurfPitch(); // Returns the actual width of the surface (not the logical width)

		// Following methods are affected by above SetRect functions
		void Fill(unsigned short color); // Fill copy rectangle with color
		void Fill(unsigned char red, unsigned char green, unsigned char blue);

		// CopyFast() only works if the source and destination are the same size and if the destination
		// surface does not have a clipper attached to it (highly limited)
		void CopyFast(GenericSurfaceObject &Source);

		// Regular copy functions - Use values set by the SetRect() functions to
		// determine source and destination copy areas
		void Copy(GenericSurfaceObject &Source);
		void Copy_Sprite(GenericSurfaceObject &Source);

		void Copy_Blend(GenericSurfaceObject &Source);
		void Copy_BlendSprite(GenericSurfaceObject &Source);

		void Copy_Scale(GenericSurfaceObject &Source);
		void Copy_ScaleSprite(GenericSurfaceObject &Source);

		void Copy_HFlip(GenericSurfaceObject &Source);
		void Copy_VFlip(GenericSurfaceObject &Source);

		void Copy_HFlipSprite(GenericSurfaceObject &Source);
		void Copy_VFlipSprite(GenericSurfaceObject &Source);

		// Tile copy functions (Surface must be tiled using SetTiled())
		void CopyTile(GenericSurfaceObject &Source, int tile);
		void CopyTile_Sprite(GenericSurfaceObject &Source, int tile);

		void CopyTile_Blend(GenericSurfaceObject &Source, int tile);
		void CopyTile_BlendSprite(GenericSurfaceObject &Source, int tile);

		void CopyTile_Scale(GenericSurfaceObject &Source, int tile);
		void CopyTile_ScaleSprite(GenericSurfaceObject &Source, int tile);

		void CopyTile_HFlip(GenericSurfaceObject &Source, int tile);
		void CopyTile_VFlip(GenericSurfaceObject &Source, int tile);

		void CopyTile_HFlipSprite(GenericSurfaceObject &Source, int tile);
		void CopyTile_VFlipSprite(GenericSurfaceObject &Source, int tile);

		unsigned short* Lock(); // Returns a pointer to the surface
		void Unlock(); // Make sure to call Unlock() as soon as possible after calling Lock()


	protected:

		// Protected data members

		LPDIRECTDRAWSURFACE4 m_pDDSurface; // m_pDDSurface represented by this class		
		int m_SurfWidth, m_SurfHeight; // Width and height of the actual m_pDDSurface		
		int m_ID; // The object ID

		// Shared (static) data members
		// DEV NOTE: These statics are not cleaned up anywhere - need to find a good place to do that

		// Temporary surface used to speed up blend copying. It is used under one of the
		// following two conditions:
		// 1) The surface is in video memory, in which case it is needed to speed up blend copying
		// 2) The surface has a clipper attached to it, in which case it is needed to ensure that
		//    we do not overwrite the non-clippable regions during blend copy
		static LPDIRECTDRAWSURFACE4 m_pDDTempSurface;		
		static DDSURFACEDESC2 m_DDSurfaceDesc;	// Shared surface description structure
		static RGB16 *m_pRGB16; // Pointer to an RGB16 structure (shared across ALL surfaces)

		// Protected member functions
		void Create(int width, int height); // Called by child classes to Create the generic surface
		void Destroy(); // Destroys the object

		// Helpers to create/destroy DIRECTDRAWSURFACEs
		bool CreateSurface(tSurfType surf_type, LPDIRECTDRAWSURFACE4 *ppDDSurface, int width, int height, bool video_ram=false);
		void DestroySurface(LPDIRECTDRAWSURFACE4 *ppDDSurface);

		// Helpers to set/get the surface flag values
		void SetFlag(unsigned char flag); // Sets to flag
		void UnsetFlag(unsigned char flag); // Unsets flag
		bool GetFlag(unsigned char flag); // Returns true if flag is set, false otherwise
		
	private:

		// Private data members

		RECT m_CopyRect; // Defines area to copy
		int m_CopyRectWidth, m_CopyRectHeight; // Width and Height in copy rectangle

		RECT *m_pTileRects; // May contain rect's for each tile (if SetTiled() is called)
		int m_TileWidth, m_TileHeight; // Width and Height of tiles

		LPDIRECTDRAWCLIPPER m_pClipper; // Clipper object
		int m_NumClipRects; // Number of clipping rectangles defined

		// internally defined struct
		typedef struct
		{
			RGNDATAHEADER rdh;
			RECT Rects[MAX_CLIP_RECTS];
		} tClipList;

		tClipList *m_pClipList; // The clip list

		unsigned char m_BlendLevel; // Blend level
		unsigned short m_SpriteColor; // Sprite color
		long m_Pitch; // Use instead of m_SurfWidth when calling Lock()

		unsigned char m_SurfaceInfoFlags; // Contains information about the surface

		FontObject *m_pFontObject; // Pointer to a FontObject		

		// Private member functions
		
		void Construct(); // Constructs the object
		void Destruct(); // Destructs the object

		// Hand-coded blending function used by all the "Blend" Copy functions
		void Blend(GenericSurfaceObject &Source, bool spriteBlend, int destOffset, int sourceOffset, int width, int height);
			
		// Helpers defined inline for speed
		void AdjustDestRect(GenericSurfaceObject &Source) 
		{
			m_CopyRect.right = m_CopyRect.left + Source.m_CopyRectWidth;
			m_CopyRect.bottom = m_CopyRect.top + Source.m_CopyRectHeight;
		}

		void AdjustDestRectTile(GenericSurfaceObject &Source, int tile)
		{
			m_CopyRect.right = m_CopyRect.left + Source.m_TileWidth;
			m_CopyRect.bottom = m_CopyRect.top + Source.m_TileHeight;
		}
};



//**********************************************************************
// A derived base class that represents surfaces that can display to the 
// screen. Cannot be instantiated.
class DisplaySurfaceObject : public GenericSurfaceObject
{
	public:

		HWND GetWinHandle(); // Returns the window handle
		tDisplayMode GetDisplayMode(); // Returns the current display mode

	protected:

		// Constructor/Destructor are protected, only children can instantiate
		DisplaySurfaceObject();
		virtual ~DisplaySurfaceObject();

		// Sets the display mode and sets data members as well
		void SetDisplayMode(HWND hwnd, tDisplayMode display_mode);

		virtual void Display() = 0; // Pure virtual, must be defined in children

	private:

		// This class handles the following data
		HWND m_hwnd;
		tDisplayMode m_DisplayMode;
};


//**********************************************************************
//**********************************************************************
// Now we declare the classes you can instantiate, the bottom-level children

//**********************************************************************
// SurfaceObject represents a typical surface of any size.
// Note that it is basically the same as a GenericSurfaceObject, but placed
// lower for design purposes (don't want a public Create())
class SurfaceObject : public GenericSurfaceObject
{
	public:
		// Constructor/Destructor
		SurfaceObject();
		virtual ~SurfaceObject();

		// Use these functions to create/destroy the object so that it
		// can be re-used
		void Create(int width, int height);
		void Destroy();
};

//**********************************************************************
// PrimarySurfaceObject is a singleton class that represents the screen
// (or window) that is displayed.
class PrimarySurfaceObject : public DisplaySurfaceObject
{
	public:

		// Use Instance() to get the single instance of this object
		static PrimarySurfaceObject* Instance();
		static void DestroyInstance(); // Destroys the instance
		static bool IsInstantiated(); // Returns true if this object was instatiated

		// Use any of the following to create the primary surface
		HWND Create(HINSTANCE hInstance, LPCSTR caption, tDisplayMode display_mode=FULLSCREEN);
		void Create(WindowObject *window_object);
		bool IsCreated(); // Returns true if the object has been created		
		
		void SwitchMode(tDisplayMode new_mode); // Changes the object to suit the new mode
		tDisplayMode ToggleDisplayMode(); // Use this function to switch between WINDOWED and FULLSCREEN mode

		void Display(); // The most important function (must be defined)

		WindowObject* GetWindowObject(); // Returns the WindowObject or NULL if non-existant

		// Functions called by GSWinUpdate() (defined in gs_helper.h)
		void LostWinFocus(); // Called when the window loses focus
		bool GotWinFocus(); // Called when the window regains the focus

		// Public destructor
		virtual ~PrimarySurfaceObject();

	private:

		// Singelton stuff
		PrimarySurfaceObject(); // Private constructor to implement singleton
		static PrimarySurfaceObject *m_pPrimarySurfaceObject; // Will point to the single instance

		// Private member functions


		// Private data members
		LPDIRECTDRAWSURFACE4 m_pDDPrimarySurface;
		WindowObject *m_pWinObject;
		bool m_CreatedWindowObject; // True if we create our own WindowObject, false if it's passed in

		int m_TopOffset, m_LeftOffset; // Top/Left offset of client screen from window corner

		bool m_SwitchingMode; // True if we're in the process of switching screen mode
};

//**********************************************************************
// WindowSurfaceObject can be used to represent a window. It is slower than PrimarySurfaceObject,
// but it can (and should) be used to wrap MFC windows for either SDI or MDI applications.
class WindowSurfaceObject : public DisplaySurfaceObject
{
	public:
		// Constructor/Destructor
		WindowSurfaceObject();
		~WindowSurfaceObject();

		void Create(HWND hwnd, HDC hdc); // Creates the object using the size set for DirectDrawObject
		void Create(HWND hwnd, HDC hdc, int width, int height); // Creates the object using the given size


		void Display(); // The most important function (must be defined)

	private:

		HDC m_WindowHDC;	// Handle to device context of the window surface
		HDC m_DDSurfaceHDC;	// Handle to the device context of the DirectDraw surface
};

#endif
