// gs_winstandard.h

// Contains the standard Windows functions and classes needed to create/destroy windows.

#ifndef GS_WINSTANDARD_H
#define GS_WINSTANDARD_H

#include <windows.h>
#include <cstdio>

#include "gs_common.h"

// Typedef that defines the current mouse state
typedef struct MOUSE_STATE_STRUCT
{
	int x, y;
	bool leftDown, leftUp;
	bool rightDown, rightUp;
} tMouseState;


// The default Windows callback procedure. You don't need to make one unless you want to.
LRESULT CALLBACK GenericWindowProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam);

//**********************************************************************
// MouseObject allows the user to interact with the mouse (singleton)
class MouseObject
{
	friend LRESULT CALLBACK GenericWindowProc(HWND, UINT, WPARAM, LPARAM);

	public:
		static MouseObject* Instance(); // Returns the single instance of this object
		static void DestroyInstance(); // Destroys the instance
		static bool IsInstantiated(); // Returns true if this singleton has been instantiated		

		void ShowMouse(bool show);

		void GetPos(int &x, int &y);
		int GetX();
		int GetY();

		bool IsLeftDown();
		bool IsLeftUp();

		bool IsRightDown();
		bool IsRightUp();

	private:
		MouseObject();	// Private constructor
		~MouseObject();	// Private destructor

		static MouseObject *m_pMouseObject; // The instance pointer

		static tMouseState m_MouseState; // The current mouse state
		int m_ID; // The object ID
};


class PrimarySurfaceObject; // Declaration for friendship

//**********************************************************************
// WindowObject handles the creation and managament of a window
class WindowObject
{
	friend PrimarySurfaceObject; // Let this object see WindowObject's privates

	public:
		WindowObject(HINSTANCE hInstance, WNDPROC wndProc = GenericWindowProc, LPCSTR className = "MainWin"); // Constructor
		~WindowObject(); // Destructor

		// Call these facilitators before calling Create()
		void EnableDoubleClicks(); // Enables double-clicks
		void UseMenu(LPCSTR MenuName); // Loads a menu for the window

		// Creates the window and returns it's handle
		HWND Create(LPCSTR caption = "Window", int width = CW_USEDEFAULT, int height = CW_USEDEFAULT, HWND parent_hwnd = NULL, tDisplayMode screen_mode = FULLSCREEN); 

		tDisplayMode GetDisplayMode();			// Returns the current display mode set on the object

		// Call these facilitators after calling Create()
		HWND GetWinHandle();	// Returns the window handle (if needed by other functions)
		DWORD GetWindowStyle();		// Returns the window style
		DWORD GetWindowStyleEx();	// Returns the extended window style

		HACCEL LoadAccelTable(LPCSTR lpMenuName);	// Loads and returns handle to an Accelerator Table
		void ShowMouse(bool show);					// If show=false, mouse is hidden
		bool IsMouseShowing();						// Returns true if mouse is showing

		void Show(UINT cmdShow = SW_SHOW); // Displays the window

		void HandleWinMsgs();			// Call at the end of WinMain() to handle windows messages
		void HandleWinMsgs(HACCEL hAccel);	// Call this if using key accelerators
		void HandleWinMsgsFast();		// Must be called within a loop (great for games) - BUGGY!!!

		int GetWidth();		// Returns the width of the window
		int GetHeight();	// Returns the height of the window

		int QuitStatus();	// Return this function as last line of WinMain()

	private:

		void SwitchMode(tDisplayMode new_mode); // Changes the window to suit the new mode

		// Adjusts the window size so that client area is width*height, optinally fills
		// in the top and left offsets to the client area and resizes the window
		void AdjustSize(int *top_offset=NULL, int *left_offset=NULL, bool resize=false);


		HWND hwnd;								// Handle to window
		WNDCLASSEX wndClassEx;					// Window class object (Ex = Small icons, etc.)
		MSG msg;								// To store windows messages
		int winWidth, winHeight;				// width and height of the window
		int clientWidth, clientHeight;			// width and height of client area of window
		DWORD m_WindowStyle, m_WindowStyleEx;	// The window styles
		tDisplayMode m_DisplayMode;				// The current display mode
		bool m_ShowMouse;						// True if the mouse is being shown, false otherwise
		int m_ID;								// The object ID
};

#endif
