// winmain.cpp

// Programmer: Antonio Maiorano
// Compiled with: Visual C++ 6.0
// Using the Guardian Soft DirectX Wrapper Library

// This demo shows off the functionality of the Guardian Soft DirectX Library

// Make sure your Visual Studio environment is setup properly:
//
// 1) Add the path to gsdxlib\include to the Includes
// 2) Add the path to gsdxlib\lib to the Libraries
//
// See the help for more details.

// The only file you need to include to use the library
#include <gs_lib.h>


// Application defines
#define SCREEN_WIDTH			640
#define SCREEN_HEIGHT 			480
#define SCREEN_WIDTHxHEIGHT		307200
#define BLACK 					0
#define WHITE 					65535

// Globals
SurfaceObject TextSurface;	// Used to write text to (global for functions)


// Simple function used to display Screen
inline void ShowScreen(void)
{
	static PrimarySurfaceObject *pPrimSurf = PrimarySurfaceObject::Instance();

	pPrimSurf->Display();
}

// Overloaded function that displays Source to Screen
inline void ShowScreen(SurfaceObject &Source)
{
	static PrimarySurfaceObject *pPrimSurf = PrimarySurfaceObject::Instance();

	// Assumes that Source is same size as Screen
	Source.ResetRect();
	pPrimSurf->ResetRect();

	pPrimSurf->Copy(Source);
	pPrimSurf->Display();
}

inline void ShowScreenAndWait(SurfaceObject &Source)
{
	static KeyboardObject *pKeyboard = KeyboardObject::Instance();

	while (pKeyboard->GetKeyUp() != DIK_RETURN)
	{
		ShowScreen(Source); // Show the surface to the screen
		GSWinUpdate(); // So user can Alt+Tab
	}
}

// Clears TextSurface to black, writes message to it, displays it and waits for <ENTER>
inline void ShowTextAndWait(char message[], int x, int y)
{
	static KeyboardObject *pKeyboard = KeyboardObject::Instance();

	TextSurface.ResetRect();
	TextSurface.Fill(BLACK);
	TextSurface.Text(x, y, message);

	// This idle loop, unfortunately, must constantly redraw TextSurface
	// to the Screen because otherwise, Windows will eventually repaint the
	// window, and we won't see the TextSurface anymore. This is true as long
	// as you call GSWinUpdate(). Of course, you can choose not to call GSWinUpdate(),
	// but then your game won't work with Windows (no multitasking).
	while (pKeyboard->GetKeyUp() != DIK_RETURN)
	{
		ShowScreen(TextSurface); // Show the surface to the screen
		GSWinUpdate(); // So user can Alt+Tab
	}
}


//**********************************************************************
int WINAPI WinMain(HINSTANCE hThisInstance, HINSTANCE hPrevInstance, LPSTR lpszArgs, int nWinMode)
{
	bool FullScreen;

	if (MessageBox(NULL, "Run in Fullscreen?", "Screen Mode", MB_YESNO) == IDYES)
		FullScreen = true;
	else
		FullScreen = false;

	int i = 0, j = 0, k = 0; // Counters used throughout the program

	//////////////////////////////////////////////////////////////////////
	// Initialize all objects

	// Singleton pointers
	DirectDrawObject		*pDDrawObject	= NULL;
	PrimarySurfaceObject	*pPrimSurf		= NULL;
	DirectInputObject		*pDInputObject	= NULL;
	DirectSoundObject		*pDSoundObject	= NULL;
	DirectMusicObject		*pDMusicObject	= NULL;
	KeyboardObject			*pKeyboard		= NULL;
	MouseObject				*pMouse			= NULL;

	// If there's a second monitor, then lets use that monitor
	DisplayDeviceEnumerator deviceEnum;
/*
	if (FullScreen && deviceEnum.GetNumDevices() > 1)
		deviceEnum.SetDevice(2); // 2 = second monitor
	else*/
		deviceEnum.SetDevice(0); // 0 = default monitor (same as 1)

//deviceEnum.SetDevice(0);

	// Initialize the graphics system using a helper function
	GSInitGraphics(hThisInstance, SCREEN_WIDTH, SCREEN_HEIGHT, FullScreen? FULLSCREEN : WINDOWED,
				   "Guardian Soft DirectX Wrapper Library", "Graphics", &deviceEnum, &pDDrawObject, &pPrimSurf);

	HWND hwnd = pPrimSurf->GetWinHandle(); // Get the window handle

	// Show black with a message while loading (nicer than seeing an ugly white window!)...
	pPrimSurf->Fill(0);
	pPrimSurf->Text(300, 230, "Loading");
	ShowScreen();
	
	// Initialize the input system using a helper function
	GSInitInput(hThisInstance, hwnd, &pDInputObject, &pKeyboard);

	// Initialize the audio system using a helper function
	GSInitAudio(hwnd, "Waves", "Midis", &pDSoundObject, &pDMusicObject);

	// Create the global TextSurface, create a FontObject, then make the TextSurface
	// use the FontObject
	TextSurface.Create(SCREEN_WIDTH, SCREEN_HEIGHT);
	TextSurface.Fill(BLACK);

	// Here we create a local FontObject. Note that if this object were to go out of scope,
	// then if you attempt to write text to a surface that uses this FontObject, the program
	// would crash and burn! Each surface that uses a FontObject simply keeps a pointer to
	// the existing object (as oppose to a local copy which would waste memory). The best
	// idea is to allocate FontObjects using 'new' and deleting them at the end of the program.
	FontObject TestFont;
	TestFont.Create(FONT_TIMES_NEW_ROMAN, 20);

	// Make the Screen and TextSurface surfaces use the TestFont
	pPrimSurf->SetFontObject(TestFont);
	TextSurface.SetFontObject(TestFont);

	// Note that you can change the attributes of the font anytime and
	// the next text output to a surface will reflect the change
	TestFont.SetAttributes(true, false, false, false); // Bold, Italic, Underline, and Strikeout
	TestFont.SetBkTransparent(true); // So it doesn't draw the background rectangle
	TestFont.SetColors(TEXT_WHITE, TEXT_BLUE); // Set the foreground and background font colors

	// Now create all the surfaces and load graphics onto them

	SurfaceObject Kaelin;	// To load Kaelin background
	SurfaceObject Fish;		// To load the ugly fish picture
	SurfaceObject Monster;	// To load the unused ZeldaPC monster
	SurfaceObject MidiTest;	// To load the Midi and Sound test background
	SurfaceObject Temp;		// Temporary surface

	Kaelin.Create(SCREEN_WIDTH, SCREEN_HEIGHT);
	Fish.Create(SCREEN_WIDTH, SCREEN_HEIGHT);
	Monster.Create(96, 32);

	// A special feature of the LoadRAW function is specifying an offset into the RAW file from
	// which to load from (optional parameter). This is useful if you want to place many graphics
	// into one RAW file.

	// Kaelin is the first image, so there is no need to specify an offset
	Kaelin.LoadRAW("KaelinAndFish.raw", SCREEN_WIDTH, SCREEN_HEIGHT);

	// The fish is the second image in the file, so specify an offset of one screen size
	Fish.LoadRAW("KaelinAndFish.raw", SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTHxHEIGHT);

	// You can also load from BMP files, and since the size is specified in the file,
	// we don't need to pass it to LoadBMP()
	Monster.LoadBMP("monsters.bmp");
	Monster.SetTiled(32, 32); // Set the surface to tiled 32x32

	// End of Initialize all objects
	//////////////////////////////////////////////////////////////////////

//pPrimSurf->GetWindowObject()->ShowMouse(true);

	/*
	char string[255];
	pMouse->ShowMouse(true);
	while (1)
	{
		if (pKeyboard->KeyDown(DIK_ESCAPE))
			exit(1);

		if (pMouse->IsLeftDown())
		{
			sprintf(string, "Mouse position: (%i,%i)", pMouse->GetX(), pMouse->GetY());
			::MessageBox(NULL, string, "Mouse position", MB_OK);
		}

		if (pMouse->IsRightDown())
		{
			while (!pMouse->IsRightUp())
			{
				// This is ugly, but we need to call this while waiting for the
				// right mouse button to come up (aargh!) Oh well, at least it works
				GSWinUpdate();
			}
			::MessageBox(NULL, "Right mouse button just went up!", "BLAH", MB_OK);
		}

		// Need to call GSWinUpdate() for mouse support (otherwise the mouse position
		// won't be updated)
		GSWinUpdate(); // So user can Alt+Tab
	}
	*/

	/*
	//////////////////////////////////////////////////////////////////////
	// Test the clipper

	pPrimSurf->Fill(0, 0, 255);
	pPrimSurf->Display();
	pPrimSurf->Fill(0, 0, 255);

	// Create the clipping rectangles for Screen and then attach it to Screen

	pPrimSurf->AddClipRect(10, 10, 315, 235);
	pPrimSurf->AddClipRect(325, 10, 630, 235);
	pPrimSurf->AddClipRect(10, 245, 315, 470);
	pPrimSurf->AddClipRect(325, 245, 630, 470);

	pPrimSurf->AttachClipper();
 

	// pPrimSurf->DetachClipper();


	// End of test the clipper
	//////////////////////////////////////////////////////////////////////
	*/

	//////////////////////////////////////////////////////////////////////
	// Scaling a tile sprite

	#ifdef GS_DEBUG
		DebugMsg(NULL, "winmain.cpp", "Scaling a tile sprite...");
	#endif

	ShowTextAndWait("Scaling a tile sprite... (Press Enter)", 10, 10);

	Kaelin.ResetRect(); // Set source rectangle

	for (i=0; i<200; ++i)
	{
		// Copy Kaelin to the Screen
		pPrimSurf->ResetRect();
		pPrimSurf->Copy(Kaelin);

		// Set the destination rectangle (Screen) to scale the monster tile (Monster)
		pPrimSurf->SetRectSize(0, 0, i, i);
		pPrimSurf->CopyTile_ScaleSprite(Monster, 1);

		ShowScreen(); // Display the Screen

		// Handle standard windows processing, making DirectX and Windows work in harmony
		GSWinUpdate(); // So user can Alt+Tab
	}

	// End of Scaling a tile sprite
	//////////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////////
	// Scaling a regular sprite

	#ifdef GS_DEBUG
		DebugMsg(NULL, "winmain.cpp", "Scaling a regular sprite - cooler effect...");
	#endif

	ShowTextAndWait("Scaling a regular sprite - cooler effect... (Press Enter)", 10, 10);

	Fish.SetRectSize(100, 100, 150, 150);
	for (i=316, j=8; i>=0; i-=2, j+=4)
	{
		pPrimSurf->ResetRect();
		pPrimSurf->Copy(Kaelin);

		pPrimSurf->SetRectSize(i, 0, j, SCREEN_HEIGHT);
		pPrimSurf->Copy_ScaleSprite(Fish);

		ShowScreen();

		// Handle standard windows processing, making DirectX and Windows work in harmony
		GSWinUpdate(); // So user can Alt+Tab
	}

	// End of Scaling a regular sprite
	//////////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////////
	// Scaling a non-sprite bitmap

	#ifdef GS_DEBUG
		DebugMsg(NULL, "winmain.cpp", "Scaling a bitmap (non-sprite)...");
	#endif

	ShowTextAndWait("Scaling a bitmap (non-sprite)... (Press Enter)", 10, 10);

	for (i=250; i>=0; --i)
	{
		pPrimSurf->ResetRect();
		pPrimSurf->Copy(Kaelin);

		pPrimSurf->SetRectSize(50, 50, i, i);
		pPrimSurf->Copy_Scale(Fish);

		ShowScreen();

		// Handle standard windows processing, making DirectX and Windows work in harmony
		GSWinUpdate(); // So user can Alt+Tab
	}

	// End of Scaling a non-sprite bitmap
	//////////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////////
	// Blending a sprite

	#ifdef GS_DEBUG
		DebugMsg(NULL, "winmain.cpp", "Blending a sprite...");
	#endif

	ShowTextAndWait("Blending a sprite... (Press Enter)", 10, 10);

	Fish.SetRectSize(100, 100, 300, 300);
	for (int p=0; p<=255; p+=5)
	{
		// Copy Kaelin to Screen
		pPrimSurf->ResetRect();
		pPrimSurf->Copy(Kaelin);

		// Copy Fish to Screen with blending
		Fish.SetBlendLevel(p);
		pPrimSurf->SetRectSize(20, 20, 300, 300);
		pPrimSurf->Copy_BlendSprite(Fish); // Copy Fish to Screen as a sprite with blending

		ShowScreen(); // Display the Screen

		// Handle standard windows processing, making DirectX and Windows work in harmony
		GSWinUpdate(); // So user can Alt+Tab
	}

	// End of Blending a sprite
	//////////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////////
	// Animating a blended sprite

	#ifdef GS_DEBUG
		DebugMsg(NULL, "winmain.cpp", "Animating a blended sprite...");
	#endif

	ShowTextAndWait("Animating a blended sprite... (Press Enter)", 10, 10);

	Kaelin.ResetRect(); // Set up copy rectangle of Kaelin to complete surface

	for (i=608, j=0; i>=0; --i)
	{ 	
		// Copy Kaelin to Screen at (0,0)
		pPrimSurf->ResetRect();
		pPrimSurf->Copy(Kaelin);

		// Copy tile j from Monster to Screen at (i,200)
		pPrimSurf->SetRectSize(i, 200, 32, 32);
		pPrimSurf->CopyTile_BlendSprite(Monster, j);

		ShowScreen(); // Now display Screen

		if (i%8 == 0)
			j = ++j%3;

		// Handle standard windows processing, making DirectX and Windows work in harmony
		GSWinUpdate(); // So user can Alt+Tab
	}

	// End of Animating a blended sprite
	//////////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////////
	// Scaling and flipping a sprite tile
	#ifdef GS_DEBUG
		DebugMsg(NULL, "winmain.cpp", "Scaling a tiled sprite with flipping, very cool spin effect...");
	#endif

	ShowTextAndWait("Scaling a tiled sprite with flipping, very cool spin effect... (Press Enter)", 10, 10);

	Monster.SetRect(0, 0, 32, 32);

	// Spin 20 times
	for (k=0; k<20; ++k)
	{
		// Grow
		for (i=0; i<16; ++i)
		{
			pPrimSurf->ResetRect();
			pPrimSurf->Copy(Kaelin);

			pPrimSurf->SetRect(50-i, 50, 50+i, 50+32);
			pPrimSurf->CopyTile_ScaleSprite(Monster, 0);

			ShowScreen();

		// Handle standard windows processing, making DirectX and Windows work in harmony
		GSWinUpdate(); // So user can Alt+Tab
		}

		// Shrink
		for (i=15; i>=0; --i)
		{
			pPrimSurf->ResetRect();
			pPrimSurf->Copy(Kaelin);

			pPrimSurf->SetRect(50-i, 50, 50+i, 50+32);
			pPrimSurf->CopyTile_ScaleSprite(Monster, 0);

			ShowScreen();

		// Handle standard windows processing, making DirectX and Windows work in harmony
		GSWinUpdate(); // So user can Alt+Tab
		}

		// Flip the tile on itself
		Monster.CopyTile_HFlip(Monster, 0);
	}

	// End of Scaling and flipping a sprite tile
	//////////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////////
	// Try the Academy Awards special effect fade out!

	// Here's an example of using the DebugMsg() function (defined in gs_common.h) 
	// within the code (should include the #ifdef GS_DEBUG and #endif)
	#ifdef GS_DEBUG
		DebugMsg(NULL, "winmain.cpp", "Academy Awards special effect fade out...");
	#endif

	ShowTextAndWait("Academy Awards special effect fade out... (Press Enter)", 10, 10);

	Temp.Create(300, 100);
	Temp.LoadBMP("Banderas.bmp");

	SurfaceObject Banderas;

	Banderas.Create(600, 100);
	Banderas.Fill(BLACK);
	Banderas.Copy(Temp);

	Banderas.SetSpriteColor(255, 0, 255);

	int width=300, blevel = 255;

	for (i=170; i>=17; i-=3)
	{
		pPrimSurf->ResetRect();
		pPrimSurf->Copy(Kaelin);

		// Display to screen
		pPrimSurf->SetRectSize(i, 200, width, 100);
		Banderas.SetRectSize(0, 0, width, 100);
		
		Banderas.SetBlendLevel(blevel);
		pPrimSurf->Copy_BlendSprite(Banderas);

		blevel -= 5;

		pPrimSurf->Display();

		// Stretch
		width += 6;

		Banderas.SetRectSize(0, 0, width, 100);
		Banderas.Copy_Scale(Temp);

		// Handle standard windows processing, making DirectX and Windows work in harmony
		GSWinUpdate(); // So user can Alt+Tab
	}

	pPrimSurf->ResetRect();
	pPrimSurf->Copy(Kaelin);
	pPrimSurf->Display();

	Temp.Destroy(); // Destroy Temp so we can reuse it later

	// End of Academy Awards special effect fade out
	//////////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////////
	// Show how to use the clipper
	
	#ifdef GS_DEBUG
		DebugMsg(NULL, "winmain.cpp", "A clipping example...");
	#endif

	ShowTextAndWait("A clipping example... (Press Enter)", 10, 10);

	pPrimSurf->AddClipRect(); // First we create a clipper rectangle the size of the surface
	pPrimSurf->AttachClipper(); // Now we attach the clipper to the surface

	// Create a surface larger than the screen and load a bitmap onto it
	SurfaceObject Large;
	Large.Create(700, 480);
	Large.LoadBMP("Large.bmp");


	// Now we loop by simply copying the larger surface onto the screen, showing how
	// it is automatically clipped
	pPrimSurf->SetRect(0, 0);

	for (i=0; i<SCREEN_HEIGHT; ++i)
	{
		pPrimSurf->SetRect(0, 0);
		pPrimSurf->Copy(Kaelin);

		pPrimSurf->SetRect(i, i);
		pPrimSurf->Copy(Large);

		pPrimSurf->Display();

		// Handle standard windows processing, making DirectX and Windows work in harmony
		GSWinUpdate(); // So user can Alt+Tab
	}

	// End of show how to use the clipper
	//////////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////////
	// Another clipper effect using a tiled surface

	#ifdef GS_DEBUG
		DebugMsg(NULL, "winmain.cpp", "Another clipping example using a tiled surface...");
	#endif

	ShowTextAndWait("Another clipping example using a tiled surface... (Press Enter)", 10, 10);

	// Create a temporary surface the size of the screen and attach a clipper to it
	Temp.Create(SCREEN_WIDTH, SCREEN_HEIGHT);
	Temp.AddClipRect();
	Temp.AttachClipper();

	// Create an exact copy of Kaelin, but make this one tiled into vertical strips
	SurfaceObject Kaelin2(Kaelin);

	int stripWidth = 4;
	Kaelin2.SetTiled(stripWidth, SCREEN_HEIGHT);

	// Now we loop and create a screen-splitting effect
	pPrimSurf->SetRect(0, 0);
	int yVal = 0;

	for (i=0; i<SCREEN_HEIGHT; ++i)
	{
		Temp.Fill(BLACK);

		for (j=0; j<(SCREEN_WIDTH/stripWidth); ++j)
		{
			Temp.SetRect(j*stripWidth, j%2? yVal : -yVal);
			Temp.CopyTile(Kaelin2, j);
		}

		Temp.ResetRect();
		pPrimSurf->Copy(Temp);

		pPrimSurf->Display();

		++yVal;

		// Handle standard windows processing, making DirectX and Windows work in harmony
		GSWinUpdate(); // So user can Alt+Tab
	}

	Temp.Destroy(); // Destroy Temp so we can reuse it later
	
	// End of another clipper effect using a tiled surface
	//////////////////////////////////////////////////////////////////////


	//////////////////////////////////////////////////////////////////////
	// Show off DirectSound and DirectMusic stuff!

	pPrimSurf->ResetRect();
	int TextY = 10;
	int TextYOffset = 25;

	#ifdef GS_DEBUG
		DebugMsg(NULL, "winmain.cpp", "Time to test Sound and Music...");
	#endif

	ShowTextAndWait("Time to test Sound and Music... (Press Enter)", 10, TextY);

	// Now load the waves
	char* Waves[] = {"jad0007a.wav", "jad0016a.wav", "jad0036a.wav", "jad0046a.wav", "jad0054a.wav"};
	int NumWaves = 5;

	for (i=0; i<NumWaves; ++i)
	{
		TextSurface.Text(10, TextY+=TextYOffset, "Loading \"%s\"...", Waves[i]);
		ShowScreen(TextSurface);
		pDSoundObject->AddWave(Waves[i]);
	}

	// Now load the midis
	char* Midis[] = {"airship.mid", "baronmix.mid", "bigwhale.mid", "boss_mus.mid"};
	int NumMidis = 4;

	for (i=0; i<NumMidis; ++i)
	{
		TextSurface.Text(10, TextY+=TextYOffset, "Loading \"%s\"...", Midis[i]);
		ShowScreen(TextSurface);
		pDMusicObject->AddMidi(Midis[i]);
	}

	// All done! Note that here we're not calling GSWinUpdate() which is really bad because the user
	// cannot minimize, move, or do anything with the window (this is bad Windows manners!)
	TextSurface.Text(10, TextY+=(TextYOffset+TextYOffset), "Files loaded successfully, press Enter to continue...");
	ShowScreen(TextSurface);
	pKeyboard->WaitKeyUp(DIK_RETURN);

	// Load the new background bitmap
	MidiTest.Create(SCREEN_WIDTH,SCREEN_HEIGHT);
	MidiTest.LoadRAW("miditest.raw", SCREEN_WIDTH, SCREEN_HEIGHT);

	bool Quit = false;
	int key;

	pKeyboard->FlushBuffer(); // Empty any input that might have come through

	// Main game loop would look something like this...

	while (!Quit)
	{

		// This method of handling input allows you to check if the user
		// is pressing a certain key (Shift in this case) and any other
		// key. It only responds if the user releases the key (KeyUp()).
		if (pKeyboard->KeyDown(DIK_LSHIFT))
		{
			if (pKeyboard->KeyUp(DIK_Q)) pDSoundObject->PlayWave(0, true);
			else if (pKeyboard->KeyUp(DIK_W)) pDSoundObject->PlayWave(1, true);
			else if (pKeyboard->KeyUp(DIK_E)) pDSoundObject->PlayWave(2, true);
			else if (pKeyboard->KeyUp(DIK_R)) pDSoundObject->PlayWave(3, true);
			else if (pKeyboard->KeyUp(DIK_T)) pDSoundObject->PlayWave(4, true);
		}

		// Here's another method of handling input. Here we get the key being
		// pressed and store it into an int (key). This method is especially good
		// if you want the user to type his/her name, since you don't know what
		// input to expect. Also, it's MUCH easier (and cleaner) than having a huge
		// switch() statement where each case handles one of several possible keys
		// the user might press.
		else if (key = pKeyboard->GetKeyUp())
		{
			switch (key)
			{
				// Control the midis with the number keys. If you switch from one midi
				// to another, the old midi is automatically Paused, and when you Play
				// it again, it will resume where it left off. For the midi to start
				// from the beginning, you have to Stop it.

				case DIK_1: pDMusicObject->PlayMidi(0); break;
				case DIK_2: pDMusicObject->PlayMidi(1); break;
				case DIK_3: pDMusicObject->PlayMidi(2); break;
				case DIK_4: pDMusicObject->PlayMidi(3); break;
				
				case DIK_9: pDMusicObject->PauseMidi(); break;
				case DIK_0: pDMusicObject->StopMidi(); break;

				case DIK_Q: pDSoundObject->PlayWave(0); break;
				case DIK_W: pDSoundObject->PlayWave(1); break;
				case DIK_E: pDSoundObject->PlayWave(2); break;
				case DIK_R: pDSoundObject->PlayWave(3); break;
				case DIK_T: pDSoundObject->PlayWave(4); break;

				case DIK_A: pDSoundObject->StopWave(0); break;
				case DIK_S: pDSoundObject->StopWave(1); break;
				case DIK_D: pDSoundObject->StopWave(2); break;
				case DIK_F: pDSoundObject->StopWave(3); break;
				case DIK_G: pDSoundObject->StopWave(4); break;

				case DIK_Z: pDSoundObject->MixWave(0); break;
				case DIK_X: pDSoundObject->MixWave(1); break;
				case DIK_C: pDSoundObject->MixWave(2); break; 
				case DIK_V: pDSoundObject->MixWave(3); break; 
				case DIK_B: pDSoundObject->MixWave(4); break; 

				case DIK_L: pPrimSurf->ToggleDisplayMode(); break;

				case DIK_ESCAPE: Quit = true; break;
			}
		}

		pDMusicObject->RepeatMidi(); // This makes sure that if a midi is playing and stops, it will loop

		GSWinUpdate(); // Make sure DirectX and Windows work together!
				
		ShowScreen(MidiTest); // Update the screen (typical of game loops)	
	}
	
	// End of Show off DirectSound and DirectMusic stuff!
	//////////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////////
	// Quit the application gracefully

	#ifdef GS_DEBUG
		DebugMsg(NULL, "winmain.cpp", "Quitting normally...");
	#endif

	// Although it's not necessary, it's good practice to kill all the singletons
	// because their destructors will not be called automatically
	DirectDrawObject::DestroyInstance();
	PrimarySurfaceObject::DestroyInstance();
	DirectInputObject::DestroyInstance();
	DirectSoundObject::DestroyInstance();
	DirectMusicObject::DestroyInstance();
	KeyboardObject::DestroyInstance();
	MouseObject::DestroyInstance();
	
	return 0;

	// End of Quit the application gracefully
	//////////////////////////////////////////////////////////////////////
}
