// MultiSemTest.java 
// Programmer: Antonio Maiorano (maioran@cs.concordia.ca)
// Date: Februray 10, 2002
//
// This program presents a possible implementation of
// an AND synchronization object, MultiSemaphore in this
// case. It uses class Semaphore.

import java.lang.Math.*; // For Math.random()

// From PA2
class Semaphore {
  /*private*/ int value; // Allow access within this package
  Semaphore (int value1) {
    value = value1;
  }

  public synchronized void Wait () {
    while( value <= 0 ) {
      try { wait (); }
      catch (InterruptedException e) { };
    }
    value--;
  }

  public synchronized void Signal () {
    ++value;
    notify ();
  }
}

// This class can be used to synchronize threads on
// multiple semaphores. If a thread calls WaitMulti(),
// it will return from the call ONLY if it can acquire
// all sempahores, otherwise it blocks.
class MultiSemaphore {
	Semaphore[] sems; // Semaphores to synchronize on

	// Constructor: expects array of semaphores to sync on
	MultiSemaphore(Semaphore[] sems) {
		this.sems = sems;
	}

	private boolean IsAnyZero() {
		for (int i=0; i<sems.length; ++i) {
			if (sems[i].value <= 0) {
				return true;	
			}
		}
		return false;
	}

	public synchronized void WaitMulti() {
		// If any of the semaphores equal 0, then we have
		// to wait (block) until they are available
		while ( IsAnyZero() ) {
			try { 
				wait(); 
			} catch (InterruptedException e) {}
		}
		// If we reach here, that means we can acquire all
		// the semaphores!

		// Decrement all semaphore values
		for (int i=0; i<sems.length; ++i)
			--(sems[i].value);
	}

	public synchronized void SignalMulti() {

		// Increment all semaphore values
		for (int i=0; i<sems.length; ++i)
			++(sems[i].value);
		
		notify(); // Wake up a blocked thread arbitrarily
	}
}

// Drummer is a thread that tries to play drums by acquiring
// the two drum sticks using a MultiSemaphore
class Drummer extends Thread {
	String name;
	MultiSemaphore drumSticks;

	Drummer(String name, MultiSemaphore drumSticks) {
		this.name = name;
		this.drumSticks = drumSticks;
	}

	void playDrums() {
		System.out.println(name + " is playing drums");
	}

	public void run() {
		for (int i=0; i<100; ++i) {
			System.out.println(name + " tries to get both drum sticks...");
			drumSticks.WaitMulti();

			System.out.println(name + " got both drum sticks!");
			playDrums();

			System.out.println(name + " releases both drum sticks");
			drumSticks.SignalMulti();

			if (Math.random() >= 0.5) // 50% chance
				yield();
		}
	}
}

// Main entry point
class MultiSemTest {
	public static void main(String[] args) {

		// Declare 2 drum sticks as regular Semaphores
		Semaphore drumStick1 = new Semaphore(1);
		Semaphore drumStick2 = new Semaphore(1);

		// Put them into an array
		Semaphore[] drumSticksArr = { drumStick1, drumStick2 };

		// Create a MultiSemaphore that will sync on the two drumsticks
		MultiSemaphore drumSticks = new MultiSemaphore(drumSticksArr);

		// Create Drummer threads that will share the 2 drum sticks by
		// using the MultiSemaphore
		Drummer[] drummers = new Drummer[3];
		drummers[0] = new Drummer("Tony", drumSticks);
		drummers[1] = new Drummer("Paul", drumSticks);
		drummers[2] = new Drummer("John", drumSticks);

		// Add all drummers to the ready queue
		for (int i=0; i<drummers.length; ++i)
			drummers[i].start();

		// Wait for the drummers to finish
		for (int i=0; i<drummers.length; ++i) {
			try { drummers[i].join(); }
			catch (InterruptedException e) { };
		}
	}
}

